/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: pgpDialogs.cpp,v 1.14.2.3 2001/04/25 19:00:21 hal Exp $
____________________________________________________________________________*/

#include <string.h>

#include "pgpMem.h"
#include "pgpUserInterface.h"

#include "pgpContext.h"
#include "pgpDialogs.h"
#include "pgpErrors.h"
#include "pgpKeys.h"
#include "pgpKeyServer.h"
#include "pgpOptionListPriv.h"
#include "pgpPFLPriv.h"
#include "pgpRecipientDialogCommon.h"

#if PGP_MACINTOSH
#define HAVE_TEXT_UI	0
#else
#define HAVE_TEXT_UI	1
#endif

#define elemsof(x) ((unsigned)(sizeof(x)/sizeof(*x)))

#define	kCommonAllowedOptions						\
			kPGPOptionType_ParentWindowHandle,      \
			kPGPOptionType_TextUI,      			\
			kPGPOptionType_WindowTitle,				\
			kPGPOptionType_DefaultPrivateKey

#define	kCommonAllowedPassphraseOptions				\
			kCommonAllowedOptions,					\
			kPGPOptionType_DialogOptions,			\
			kPGPOptionType_DialogPrompt,			\
			kPGPOptionType_OutputPassphrase,		\
			kPGPOptionType_MinPassphraseLength,		\
			kPGPOptionType_MinPassphraseQuality,	\
			kPGPOptionType_DialogTimeout

#define	kCommonAllowedKeyServerOptions				\
			kCommonAllowedOptions

			
CPGPDialogOptions::CPGPDialogOptions(void)
{
	mContext				= kInvalidPGPContextRef;
	mClientOptions 			= kInvalidPGPOptionListRef;
	mWindowTitle			= NULL;
	mServerList				= NULL;
	mServerCount			= 0;
	mSearchBeforeDisplay	= FALSE;
	mTextUI					= FALSE;
	mNewKeys				= NULL;
	mTLSContext				= kInvalidPGPtlsContextRef;
	mPrompt					= NULL;
	mDefaultPrivateKey		= kInvalidPGPKeyDBObjRef;
	
#if PGP_WIN32
	 mHwndParent	= NULL;
#endif
}

CPGPDialogOptions::~CPGPDialogOptions(void)
{
}
	
	PGPError
CPGPDialogOptions::GatherOptions(
	PGPContextRef 		context,
	PGPOptionListRef	optionList)
{
	PGPError	err = kPGPError_NoErr;
	
	PGPValidateContext( context );
	PGPValidateParam( pgpOptionListIsValid( optionList ) );

	mContext		= context;
	mClientOptions	= optionList;

	err = pgpFindOptionArgs( optionList,
				kPGPOptionType_WindowTitle, FALSE, "%p",
				&mWindowTitle );

	if( IsntPGPError( err ) )
	{
		err = pgpFindOptionArgs( optionList,
					kPGPOptionType_DialogPrompt, FALSE, "%p",
					&mPrompt );
	}

	if( IsntPGPError( err ) )
	{
		err = pgpFindOptionArgs( optionList,
					kPGPOptionType_DefaultPrivateKey, FALSE, "%p",
					&mDefaultPrivateKey );
	}

#if HAVE_TEXT_UI
	if( IsntPGPError( err ) )
	{
		PGPBoolean	haveOption;
		PGPUInt32	textUI;
		
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_TextUI, FALSE,
						 "%b%d", &haveOption, &textUI );
		
		if( haveOption )
		{	 
			mTextUI = ( textUI != 0 );
		}
	}
#endif

#if PGP_WIN32
	if( IsntPGPError( err ) )
	{
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_ParentWindowHandle, FALSE,
						 "%p", &mHwndParent );
	}
#endif

	if( IsntPGPError( err ) &&
		IsntPGPError( pgpCheckNetworklibAvailability() ) )
	{
		PGPOUIKeyServerParamsDesc	*desc = NULL;
		
		// Key server params are stored at this level because this is the
		// class common to all users of key servers. The legal option
		// checking will filter out illegal calls.
	
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_KeyServerUpdateParams, FALSE,
						 "%p", &desc );
		
		if( IsntNull( desc ) )
		{
			mServerList 			= desc->serverList;
			mServerCount 			= desc->serverCount;
			mTLSContext				= desc->tlsContext;
			mSearchBeforeDisplay 	= desc->searchBeforeDisplay;
			mNewKeys 				= desc->foundKeys;
		}
	}

	return( err );
}

CPGPRecipientDialogOptions::CPGPRecipientDialogOptions(void)
{
	mDialogOptions				= kInvalidPGPOptionListRef;
	mNumDefaultRecipients		= 0;
	mDefaultRecipients			= NULL;
	mDisplayMarginalValidity	= FALSE;
	mIgnoreMarginalValidity		= FALSE;
	mGroupSet					= kInvalidPGPGroupSetRef;
	mClientKeySet				= kInvalidPGPKeySetRef;
	mRecipientKeysPtr			= NULL;
	mEnforcement				= kPGPARREnforcement_None;
	mAlwaysDisplay				= FALSE;
	mAlwaysDisplayWithARRs		= FALSE;
	mRecipientCount				= NULL;
	mRecipientList				= NULL;
}

CPGPRecipientDialogOptions::~CPGPRecipientDialogOptions(void)
{
	if( IsntNull( mDefaultRecipients ) )
	{
		PGPFreeData( (void *) mDefaultRecipients );
		mDefaultRecipients = NULL;
	}
}

	PGPError
CPGPRecipientDialogOptions::GatherOptions(
	PGPContextRef 		context,
	PGPOptionListRef	optionList)
{
	PGPError	err = kPGPError_NoErr;
	
	PGPValidateContext( context );
	PGPValidateParam( pgpOptionListIsValid( optionList ) );

	err = CPGPDialogOptions::GatherOptions( context, optionList );
	if( IsntPGPError( err ) )
	{
		PGPOption	optionData;

		err = pgpSearchOptionSingle( optionList,
					kPGPOptionType_DialogOptions, &optionData );
		if( IsntPGPError( err ) &&
			optionData.type == kPGPOptionType_DialogOptions &&
			PGPOptionListRefIsValid( optionData.subOptions ) )
		{
			mDialogOptions = optionData.subOptions;
		}
	}

	if( IsntPGPError( err ) )
	{
		PGPUInt32	optionIndex = 0;
		PGPOption	optionData;
		
		// Multiple default recipient options are allowed, so we
		// loop and look for all of them.
		
		while( IsntPGPError( pgpGetIndexedOptionType( optionList,
					kPGPOptionType_DefaultRecipients, optionIndex, TRUE,
					&optionData, NULL ) ) )
		{
			PGPUInt32			numNewRecipients;
			PGPUInt32			numTotalRecipients;
			PGPRecipientSpec	*newRecipientList;
			
			numNewRecipients = optionData.valueSize /
										sizeof( PGPRecipientSpec );
			numTotalRecipients	= numNewRecipients + mNumDefaultRecipients;
			
			newRecipientList = (PGPRecipientSpec *) PGPNewData(
										PGPPeekContextMemoryMgr( context ),
										numTotalRecipients *
											sizeof( PGPRecipientSpec ), 0 );
			if( IsNull( newRecipientList ) )
			{
				err = kPGPError_OutOfMemory;
				break;
			}
			
			if( IsntNull( mDefaultRecipients ) )
			{
				pgpCopyMemory( mDefaultRecipients, newRecipientList,
						mNumDefaultRecipients * sizeof( PGPRecipientSpec ) );
				PGPFreeData( (void *) mDefaultRecipients );
				mDefaultRecipients = NULL;
			}
			
			pgpCopyMemory( 	optionData.value.asPtr,
							&newRecipientList[mNumDefaultRecipients],
							numNewRecipients * sizeof( PGPRecipientSpec ) );
			
			mNumDefaultRecipients 	= numTotalRecipients;
			mDefaultRecipients		= newRecipientList;
		
			++optionIndex;
		}
	}

	if( IsntPGPError( err ) )
	{
		PGPBoolean	haveOption;
		PGPUInt32	displayMarginalValidity;

		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_DisplayMarginalValidity, FALSE,
						 "%b%d", &haveOption, &displayMarginalValidity );
	
		if( haveOption )
		{	 
			mDisplayMarginalValidity = displayMarginalValidity;
		}
	}

	if( IsntPGPError( err ) )
	{
		PGPBoolean			haveOption;
		PGPOUIARRParamsDesc	arrOptions;

		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_ARREnforcement, FALSE,
						 "%b%d", &haveOption, &arrOptions );
	
		if( haveOption )
		{	 
			mEnforcement = (PGPAdditionalRecipientRequestEnforcement)
					arrOptions.enforcement;
			mAlwaysDisplayWithARRs = arrOptions.displayDialog;
		}
	}

	if( IsntPGPError( err ) )
	{
		PGPBoolean	haveOption;
		PGPUInt32	ignoreMarginalValidity;

		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_IgnoreMarginalValidity, FALSE,
						 "%b%d", &haveOption, &ignoreMarginalValidity );
	
		if( haveOption )
		{	 
			mIgnoreMarginalValidity = ignoreMarginalValidity;
		}
	}

	if( IsntPGPError( err ) )
	{
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_RecipientGroups, FALSE,
						 "%p", &mGroupSet );
	}

	if( IsntPGPError( err ) )
	{
		PGPOUIRecipientListDesc	*desc = NULL;

		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_RecipientList, FALSE,
						 "%p", &desc );
	
		if( IsntNull( desc ) )
		{	
			mRecipientCount = desc->recipientCount;
			mRecipientList	= desc->recipientList;
		}
	}
	
	return( err );
}

CPGPRandomDataDialogOptions::CPGPRandomDataDialogOptions(void)
{
	mNeededEntropyBits 	= 0;
}

CPGPRandomDataDialogOptions::~CPGPRandomDataDialogOptions(void)
{
}

CPGPKeyServerDialogOptions::CPGPKeyServerDialogOptions(void)
{
}

CPGPKeyServerDialogOptions::~CPGPKeyServerDialogOptions(void)
{
}


CPGPSearchKeyServerDialogOptions::CPGPSearchKeyServerDialogOptions(void)
{
	mSearchAllServers 	= FALSE;
	mServerList			= NULL;
	mServerCount		= 0;
	mFilter				= kInvalidPGPFilterRef;
	mKeyDescription[0] 	= 0;
}

CPGPSearchKeyServerDialogOptions::~CPGPSearchKeyServerDialogOptions(void)
{
	if( PGPFilterRefIsValid( mFilter ) )
		PGPFreeFilter( mFilter );
}

	PGPError
CPGPSearchKeyServerDialogOptions::NewKeyIDListSearchFilter(
	PGPContextRef	context,
	const PGPKeyID	*keyIDList,
	PGPUInt32		keyIDCount,
	PGPFilterRef	*filter)
{
	PGPError		err 		= kPGPError_NoErr;
	PGPFilterRef	outFilter 	= kInvalidPGPFilterRef;
	PGPUInt32		keyIDIndex;
	
	for( keyIDIndex = 0; keyIDIndex < keyIDCount; keyIDIndex++ )
	{
		PGPFilterRef	curFilter;
		
		err = PGPNewKeyDBObjDataFilter( context, kPGPKeyProperty_KeyID,
					&keyIDList[keyIDIndex], sizeof( keyIDList[0] ),
					kPGPMatchCriterion_Equal, &curFilter );
		if( IsntPGPError( err ) )
		{
			if( PGPFilterRefIsValid( outFilter ) )
			{
				PGPFilterRef	newFilter;
				
				err = PGPUnionFilters( curFilter, outFilter, &newFilter );
				if( IsntPGPError( err ) )
				{
					outFilter = newFilter;
				}
			}
			else
			{
				outFilter = curFilter;
			}
		}
	
		if( IsPGPError( err ) )
			break;
	}
	
	if( IsPGPError( err ) &&
		PGPFilterRefIsValid( outFilter ) )
	{
		PGPFreeFilter( outFilter );
		outFilter = kInvalidPGPFilterRef;
	}
	
	*filter = outFilter;
	
	return( err );
}

	PGPError
CPGPSearchKeyServerDialogOptions::NewKeySetSearchFilter(
	PGPContextRef	context,
	PGPKeySetRef	keySet,
	PGPFilterRef	*filter)
{
	PGPError		err 		= kPGPError_NoErr;
	PGPFilterRef	outFilter 	= kInvalidPGPFilterRef;
	PGPKeyListRef	keyList;
	
	err = PGPOrderKeySet( keySet, kPGPKeyOrdering_Any, FALSE, &keyList );
	if( IsntPGPError( err ) )
	{
		PGPKeyIterRef	iter;
		
		err = PGPNewKeyIter( keyList, &iter );
		if( IsntPGPError( err ) )
		{
			PGPKeyDBObjRef	theKey;
			
			err = PGPKeyIterNextKeyDBObj( iter, kPGPKeyDBObjType_Key, &theKey );
			while( IsntPGPError( err ) )
			{
				PGPKeyID	keyID;
				
				err = PGPGetKeyID( theKey,  &keyID );
				if( IsntPGPError( err ) )
				{
					PGPFilterRef	keyIDFilter;
					
					err = PGPNewKeyDBObjDataFilter( context, kPGPKeyProperty_KeyID,
								&keyID, sizeof( keyID ), kPGPMatchCriterion_Equal, &keyIDFilter );
					if( IsntPGPError( err ) )
					{
						if( PGPFilterRefIsValid( outFilter ) )
						{
							PGPFilterRef	newFilter;
							
							err = PGPUnionFilters( outFilter, keyIDFilter,
										&newFilter );
							if( IsntPGPError( err ) )
							{
								outFilter = newFilter;
							}
						}
						else
						{
							outFilter = keyIDFilter;
						}
					}
				}
				
				if( IsntPGPError( err ) )
					err = PGPKeyIterNextKeyDBObj( iter, kPGPKeyDBObjType_Key, &theKey );
			}
			
			if( err == kPGPError_EndOfIteration )
				err = kPGPError_NoErr;
				
			PGPFreeKeyIter( iter );
		}
		
		PGPFreeKeyList( keyList );
	}
	
	if( IsPGPError( err ) &&
		PGPFilterRefIsValid( outFilter ) )
	{
		PGPFreeFilter( outFilter );
		outFilter = kInvalidPGPFilterRef;
	}
	
	*filter = outFilter;
	
	return( err );
}

	PGPError
CPGPSearchKeyServerDialogOptions::GatherOptions(
	PGPContextRef 		context,
	PGPOptionListRef	optionList)
{
	PGPError	err = kPGPError_NoErr;
	
	PGPValidateContext( context );
	PGPValidateParam( pgpOptionListIsValid( optionList ) );

	err = CPGPKeyServerDialogOptions::GatherOptions( context, optionList );
	if( IsntPGPError( err ) )
	{
		PGPFilterRef	filter = kInvalidPGPFilterRef;
		PGPUInt32		optionIndex = 0;
		PGPOption		optionData;
		char			keyDescription[256];
		PGPUInt32		numSearchOptions = 0;
		
		// Multiple search source options are allowed, so we
		// loop and look for all of them. If there is a single
		// search key or a key list with one item, we grab a string
		// description of the item for display in the dialog.

		keyDescription[0] = 0;
		
		while( IsntPGPError( pgpGetIndexedOption( optionList, optionIndex,
					TRUE, &optionData ) ) && IsntPGPError( err ) )
		{
			PGPFilterRef	curFilter = kInvalidPGPFilterRef;

			switch( optionData.type )
			{
				case kPGPOptionType_KeyServerSearchFilter:
				{
					curFilter = optionData.value.asFilterRef;
					(void) PGPIncFilterRefCount( curFilter );

					++numSearchOptions;
					break;
				}
				
				case kPGPOptionType_KeyServerSearchKey:
				{
					PGPKeyDBObjRef	keyRef = optionData.value.asKeyDBObjRef;
					
					if( numSearchOptions == 0 )
					{
						PGPSize	len;
						
						err = PGPGetPrimaryUserIDName( keyRef, 
									keyDescription, sizeof( keyDescription ) - 1,
									&len );
						if( IsntPGPError( err ) )
						{
							keyDescription[len] = 0;
						}
					}
					
					if( IsntPGPError( err ) )
					{
						PGPKeyID	keyID;
						PGPSize		keyIDSize;
						
						err = PGPGetKeyDBObjDataProperty( keyRef,
									kPGPKeyProperty_KeyID, &keyID,
									sizeof( keyID ), &keyIDSize );
						if( IsntPGPError( err ) )
						{
							err = PGPNewKeyDBObjDataFilter( context,
										kPGPKeyProperty_KeyID, &keyID,
										sizeof( keyID ),
										kPGPMatchCriterion_Equal, &curFilter );
						}
					}
					
					++numSearchOptions;
					break;
				}
				
				case kPGPOptionType_KeyServerSearchKeyIDList:
				{
					PGPKeyID	*keyIDList;
					PGPUInt32	keyIDCount;
					
					keyIDList 	= (PGPKeyID *) optionData.value.asPtr;
					keyIDCount 	= optionData.valueSize / sizeof( *keyIDList );
					
					if( numSearchOptions == 0 && keyIDCount == 1 )
					{
						err = PGPGetKeyIDString( keyIDList,
									kPGPKeyIDString_Abbreviated,
									keyDescription );
					}

					if( IsntPGPError( err ) )
					{
						err = NewKeyIDListSearchFilter( context, keyIDList,
										keyIDCount, &curFilter );
					}
					
					++numSearchOptions;
					break;
				}

				case kPGPOptionType_KeyServerSearchKeySet:
				{
					err = NewKeySetSearchFilter( context,
								optionData.value.asKeySetRef, &curFilter );
					
					++numSearchOptions;
					break;
				}
				default:
					break;
			}
			
			if( PGPFilterRefIsValid( curFilter ) &&
				IsntPGPError( err ) )
			{
				if( PGPFilterRefIsValid( filter ) )
				{
					PGPFilterRef	newFilter;
					
					err = PGPUnionFilters( filter, curFilter, &newFilter );
					if( IsntPGPError( err ) )
					{
						filter = newFilter;
					}
				}
				else
				{
					filter = curFilter;
				}
			}
			
			++optionIndex;
		}
		
		if( IsntPGPError( err ) )
		{
			if( numSearchOptions == 0 )
			{
				pgpDebugMsg( "No key server subjects found" );
				err = kPGPError_BadParams;
			}
			else
			{
				pgpAssert( PGPFilterRefIsValid( filter ) );
					
				mFilter = filter;
			
				if( numSearchOptions == 1 && keyDescription[0] != 0 )
					strcpy( mKeyDescription, keyDescription );
			}
		}
		else
		{
			if( PGPFilterRefIsValid( filter ) )
				PGPFreeFilter( filter );
		}
	}

	return( err );
}

CPGPSendToKeyServerDialogOptions::CPGPSendToKeyServerDialogOptions(void)
{
	mKeysToSend	= kInvalidPGPKeySetRef;
	mFailedKeys	= NULL;
}

CPGPSendToKeyServerDialogOptions::~CPGPSendToKeyServerDialogOptions(void)
{
}

CPGPPassphraseDialogOptions::CPGPPassphraseDialogOptions(void)
{
	mPassphrasePtr			= NULL;
	mDialogOptions			= kInvalidPGPOptionListRef;
	mMinPassphraseLength	= 0;
	mMinPassphraseQuality	= 0;
	mDialogTimeout			= 0;
}

CPGPPassphraseDialogOptions::~CPGPPassphraseDialogOptions(void)
{
}

	PGPError
CPGPPassphraseDialogOptions::GatherOptions(
	PGPContextRef 		context,
	PGPOptionListRef	optionList)
{
	PGPError	err = kPGPError_NoErr;
	
	PGPValidateContext( context );
	PGPValidateParam( pgpOptionListIsValid( optionList ) );

	err = CPGPDialogOptions::GatherOptions( context, optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_OutputPassphrase, TRUE,
						 "%p", &mPassphrasePtr );
	}

	if( IsntPGPError( err ) )
	{
		PGPOption	optionData;
		
		err = pgpSearchOptionSingle( optionList,
					kPGPOptionType_DialogOptions, &optionData );
		if( IsntPGPError( err ) &&
			optionData.type == kPGPOptionType_DialogOptions &&
			PGPOptionListRefIsValid( optionData.subOptions ) )
		{
			mDialogOptions = optionData.subOptions;
		}
	}

	if( IsntPGPError( err ) )
	{
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_MinPassphraseLength, FALSE,
						 "%d", &mMinPassphraseLength );
	}

	if( IsntPGPError( err ) )
	{
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_MinPassphraseQuality, FALSE,
						 "%d", &mMinPassphraseQuality );
	}

	if( IsntPGPError( err ) )
	{
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_DialogTimeout, FALSE,
						 "%d", &mDialogTimeout );
	}

	return( err );
}

CPGPConfirmationPassphraseDialogOptions::
	CPGPConfirmationPassphraseDialogOptions(void)
{
	mShowPassphraseQuality = TRUE;
}

CPGPConfirmationPassphraseDialogOptions::
	~CPGPConfirmationPassphraseDialogOptions(void)
{
}

	PGPError
CPGPConfirmationPassphraseDialogOptions::GatherOptions(
	PGPContextRef 		context,
	PGPOptionListRef	optionList)
{
	PGPError	err;
	
	PGPValidateContext( context );
	PGPValidateParam( pgpOptionListIsValid( optionList ) );
	
	err = CPGPPassphraseDialogOptions::GatherOptions( context, optionList );
	if( IsntPGPError( err ) )
	{
		PGPBoolean	haveOption;
		PGPUInt32	showPassphraseQuality;

		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_ShowPassphraseQuality, FALSE,
						 "%b%d", &haveOption, &showPassphraseQuality );
	
		if( haveOption )
		{	 
			mShowPassphraseQuality = ( showPassphraseQuality != 0 );
		}
	}
	
	return( err );
}

CPGPKeyPassphraseDialogOptions::CPGPKeyPassphraseDialogOptions(void)
{
	mVerifyPassphrase 	= TRUE;
	mDefaultKey			= kInvalidPGPKeyDBObjRef;
	mCache				= FALSE;
	mCacheGlobal		= TRUE;
	mCacheTimeout		= 0;
}

CPGPKeyPassphraseDialogOptions::~CPGPKeyPassphraseDialogOptions(void)
{
}

	PGPError
CPGPKeyPassphraseDialogOptions::GatherOptions(
	PGPContextRef 		context,
	PGPOptionListRef	optionList)
{
	PGPError	err;
	PGPBoolean	haveOption;
	
	PGPValidateContext( context );
	PGPValidateParam( pgpOptionListIsValid( optionList ) );

	err = CPGPPassphraseDialogOptions::GatherOptions( context, optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_DefaultKey, FALSE,
						 "%p", &mDefaultKey );
	}

	if( IsntPGPError( err ) )
	{
		PGPUInt32	verifyPassphrase;
		
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_VerifyPassphrase, FALSE,
						 "%b%d", &haveOption, &verifyPassphrase );
		
		if( haveOption )
		{	 
			mVerifyPassphrase = ( verifyPassphrase != 0 );
		}
	}

	if( IsntPGPError( err ) )
	{
		err = pgpFindOptionArgs( optionList,
				kPGPOptionType_CachePassphrase, FALSE,
				"%b%d%b", &haveOption, &mCacheTimeout, &mCacheGlobal );
		if( haveOption )
		{
			mCache = TRUE;
		}
	}
		
	return( err );
}

CPGPKeySetPassphraseDialogOptions::CPGPKeySetPassphraseDialogOptions(void)
{
	mFindMatchingKey	= TRUE;
	mKeySet				= kInvalidPGPKeySetRef;
	mPassphraseKeyPtr	= NULL;
}

CPGPKeySetPassphraseDialogOptions::~CPGPKeySetPassphraseDialogOptions(void)
{
}

	PGPError
CPGPKeySetPassphraseDialogOptions::GatherOptions(
	PGPContextRef 		context,
	PGPOptionListRef	optionList)
{
	PGPError	err;
	
	PGPValidateContext( context );
	PGPValidateParam( pgpOptionListIsValid( optionList ) );

	err = CPGPKeyPassphraseDialogOptions::GatherOptions(context, optionList);
	if( IsntPGPError( err ) )
	{
		PGPBoolean	haveOption;
		PGPUInt32	findMatchingKey;
		
		err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_FindMatchingKey, FALSE,
						 "%b%d", &haveOption, &findMatchingKey );
		
		if( haveOption )
		{	 
			mFindMatchingKey = ( findMatchingKey != 0 );
		}
	}
		
	return( err );
}

CPGPSigningPassphraseDialogOptions::CPGPSigningPassphraseDialogOptions(void)
{
}

CPGPSigningPassphraseDialogOptions::~CPGPSigningPassphraseDialogOptions(void)
{
}

CPGPDecryptionPassphraseDialogOptions::
	CPGPDecryptionPassphraseDialogOptions(void)
{
	mMissingKeyIDList 	= NULL;
	mMissingKeyIDCount	= 0;
}

CPGPDecryptionPassphraseDialogOptions::
	~CPGPDecryptionPassphraseDialogOptions(void)
{
	if( IsntNull( mMissingKeyIDList ) )
		PGPFreeData( mMissingKeyIDList );
}

	PGPError
CPGPDecryptionPassphraseDialogOptions::RemoveFoundKeys(PGPKeySetRef keySet)
{
	PGPInt32	keyIDIndex;
	PGPError	err 	= kPGPError_NoErr;
	PGPKeyDBRef	keyDB	= PGPPeekKeySetKeyDB( keySet );
	
	for( keyIDIndex = mMissingKeyIDCount - 1; keyIDIndex >= 0; keyIDIndex-- )
	{
		PGPKeyDBObjRef	keyObj;
		
		if( IsntPGPError( PGPFindKeyByKeyID( keyDB,
					&mMissingKeyIDList[keyIDIndex], &keyObj ) ) &&
			PGPKeySetIsMember( keyObj, keySet ) )
		{
			PGPUInt32	keysToMove;
			
			keysToMove = mMissingKeyIDCount - keyIDIndex - 1;
			if( keysToMove != 0 )
			{
				pgpCopyMemory( &mMissingKeyIDList[keyIDIndex + 1],
						&mMissingKeyIDList[keyIDIndex],
						keysToMove * sizeof( mMissingKeyIDList[0] ) );
			}
			
			--mMissingKeyIDCount;
		}
	}
	
	return( err );
}

	PGPError
CPGPDecryptionPassphraseDialogOptions::GatherMissingKeys()
{
	PGPError	err = kPGPError_NoErr;

	if( IsntNull( mMissingKeyIDList ) )
	{
		PGPFreeData( mMissingKeyIDList );
		
		mMissingKeyIDList 	= NULL;
		mMissingKeyIDCount	= 0;
	}

	if( IsntNull( mKeyIDList ) && mKeyIDCount != 0 )
	{
		PGPSize	dataSize = mKeyIDCount * sizeof( mKeyIDList[0] );
		
		/* Assume all keys are missing. Make a copy of the key ID list */
		
		mMissingKeyIDList = (PGPKeyID *) PGPNewData(
										PGPPeekContextMemoryMgr( mContext ),
										dataSize, kPGPMemoryMgrFlags_Clear );
		if( IsntNull( mMissingKeyIDList ) )
		{
			pgpCopyMemory( mKeyIDList, mMissingKeyIDList, dataSize );
			mMissingKeyIDCount = mKeyIDCount;
			
			err = RemoveFoundKeys( mKeySet );
			if( IsntPGPError( err ) &&
				IsntNull( mNewKeys ) &&
				PGPKeyDBRefIsValid( *mNewKeys ) )
			{
				PGPKeySetRef	keySet;
				
				err = PGPNewKeySet( *mNewKeys, &keySet );
				if( IsntPGPError( err ) )
				{
					err = RemoveFoundKeys( keySet );
					
					PGPFreeKeySet( keySet );
				}
			}
			
			if( IsPGPError( err ) || mMissingKeyIDCount == 0 )
			{
				PGPFreeData( mMissingKeyIDList );
				mMissingKeyIDList 	= NULL;
				mMissingKeyIDCount 	= 0;
			}
		}
		else
		{
			err = kPGPError_OutOfMemory;
		}
	}
	
	return( err );
}

	PGPError
CPGPDecryptionPassphraseDialogOptions::SearchForMissingKeys(
	void		*hwndParent,
	PGPBoolean	*foundNewKeys)
{
	PGPError	err = kPGPError_NoErr;
	
	(void) hwndParent;
	
	pgpAssert( IsntNull( foundNewKeys ) );
	*foundNewKeys = FALSE;

	if( IsntNull( mServerList ) &&
		IsntNull( mMissingKeyIDList ) &&
		IsntNull( mNewKeys ) )
	{
		pgpAssert( ! PGPKeyDBRefIsValid( *mNewKeys ) );
		
		err = PGPNewKeyDB( mContext, mNewKeys );
		if( IsntPGPError( err ) )
		{
			PGPUInt32	serverIndex;
			PGPUInt32	numNewKeys = 0;
			
			for( serverIndex = 0; serverIndex < mServerCount; serverIndex++ )
			{
				if( IsntNull( mMissingKeyIDList ) )
				{
					PGPKeyServerProtocol	serverProtocol;
					const PGPKeyServerSpec	*serverSpec;
					
					serverSpec = &mServerList[serverIndex];
					
					err = PGPGetKeyServerProtocol( serverSpec->server,
								&serverProtocol );
					if( IsntPGPError( err ) &&
						serverProtocol != kPGPKeyServerProtocol_HTTP &&
						serverProtocol != kPGPKeyServerProtocol_HTTPS )
					{
						PGPFilterRef	filter = kInvalidPGPFilterRef;
						PGPUInt32		keyIndex;
						
						for( keyIndex = 0; keyIndex < mMissingKeyIDCount;
								keyIndex++ )
						{
							PGPFilterRef	curFilter;
							
							err = PGPNewKeyDBObjDataFilter( mContext,
										kPGPSubKeyProperty_KeyID,
										&mMissingKeyIDList[keyIndex],
										sizeof( mMissingKeyIDList[keyIndex] ),
										kPGPMatchCriterion_Equal,
										&curFilter );
							if( IsntPGPError( err ) )
							{
								if( PGPFilterRefIsValid( filter ) )
								{
									PGPFilterRef	newFilter;
									
									err = PGPUnionFilters( curFilter, filter,
												&newFilter );
									if( IsntPGPError( err ) )
									{
										filter = newFilter;
									}
								}
								else
								{
									filter = curFilter;
								}
							}
							
							if( IsPGPError( err ) )
								break;
						}
						
						if( IsntPGPError( err ) )
						{
							PGPKeyDBRef	foundKeys = kInvalidPGPKeyDBRef;
							
							err = PGPSearchKeyServerDialog( mContext, 1,
									serverSpec, mTLSContext, TRUE, &foundKeys,
									PGPOUIKeyServerSearchFilter( mContext,
										filter ),
			#if PGP_WIN32
									PGPOUIParentWindowHandle( mContext,
										(HWND) hwndParent),
			#endif
									PGPOLastOption( mContext ) );
							if( IsntPGPError( err ) &&
								PGPKeyDBRefIsValid( foundKeys ) )
							{
								PGPKeySetRef	foundKeySet;
								
								err = PGPNewKeySet( foundKeys, &foundKeySet );
								if( IsntPGPError( err ) )
								{
									err = PGPCopyKeys( foundKeySet, *mNewKeys, NULL );
									if( IsntPGPError( err ) )
									{
										err = GatherMissingKeys();
									}
									
									PGPFreeKeySet( foundKeySet );
								}
								
								PGPFreeKeyDB( foundKeys );
							}
						}
						
						if( PGPFilterRefIsValid( filter ) )
							PGPFreeFilter( filter );
					}
					
				} 
			
				if( IsPGPError( err ) )
					break;
			}
			
			(void) PGPCountKeysInKeyDB( *mNewKeys, &numNewKeys );
			if( numNewKeys != 0 )
			{
				*foundNewKeys = TRUE;
			}
			else
			{
				PGPFreeKeyDB( *mNewKeys );
				*mNewKeys = kInvalidPGPKeyDBRef;
			}
		}
	}
	
	return( err );
}

CPGPOptionsDialogOptions::CPGPOptionsDialogOptions(void)
{
}

CPGPOptionsDialogOptions::~CPGPOptionsDialogOptions(void)
{
}

static const PGPOptionType sPassphraseOptionSet[] =
{
	kCommonAllowedPassphraseOptions
};

	static PGPError
pgpPassphraseDialog(
	PGPContextRef		context,
	PGPOptionListRef 	optionList)
{
	PGPError	err;

	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( pgpOptionListIsValid( optionList ) );
	
	err = pgpGetOptionListError( optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpCheckOptionsInSet( optionList, sPassphraseOptionSet,
					elemsof( sPassphraseOptionSet ) );
		if( IsntPGPError( err ) )
		{
			CPGPPassphraseDialogOptions	options;
			
			err = options.GatherOptions( context, optionList );
			if( IsntPGPError( err ) )
			{
				err = pgpPassphraseDialogPlatform( context, &options );
			}
		}
	}
	
	return( err );
}

	PGPError
PGPPassphraseDialog(
	PGPContextRef		context,
	PGPOptionListRef	firstOption,
	...)
{
	PGPError	err = kPGPError_NoErr;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	
	if( pgpContextIsValid( context ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs(context, FALSE, firstOption, args);
		va_end( args );
	
		err = pgpPassphraseDialog( context, optionList );
	
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	}
	
	return( err );
}

	PGPError
PGPConventionalDecryptionPassphraseDialog(
	PGPContextRef		context,
	PGPOptionListRef	firstOption,
	...)
{
	PGPError	err = kPGPError_NoErr;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	
	if( pgpContextIsValid( context ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs(context, FALSE, firstOption, args);
		va_end( args );
	
		err = pgpPassphraseDialog( context, optionList );
	
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	}
	
	return( err );
}

static const PGPOptionType sOptionsOptionSet[] =
{
	kCommonAllowedOptions,
	kPGPOptionType_DialogPrompt,
	kPGPOptionType_Checkbox,
	kPGPOptionType_PopupList
};

	static PGPError
pgpOptionsDialog(
	PGPContextRef		context,
	PGPOptionListRef	optionList)
{
	PGPError			err;
	
	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( pgpOptionListIsValid( optionList ) );
	
	err = pgpGetOptionListError( optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpCheckOptionsInSet( optionList, sOptionsOptionSet,
					elemsof( sOptionsOptionSet ) );
		if( IsntPGPError( err ) )
		{
			CPGPOptionsDialogOptions	options;
			
			err = options.GatherOptions( context, optionList );
			if( IsntPGPError( err ) )
			{
				err = pgpOptionsDialogPlatform( context, &options );
			}
		}
	}
	
	return( err );
}

	PGPError
PGPOptionsDialog(
	PGPContextRef		context,
	PGPOptionListRef	firstOption,
	...)
{
	PGPError	err;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	
	if( pgpContextIsValid( context ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs(context, FALSE, firstOption, args);
		va_end( args );

		err = pgpOptionsDialog( context, optionList );
		
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	} 
	
	return( err );
}

static const PGPOptionType sKeyOptionSet[] =
{
	kCommonAllowedPassphraseOptions,
	kPGPOptionType_VerifyPassphrase,
	kPGPOptionType_CachePassphrase
};

	static PGPError
pgpKeyPassphraseDialog(
	PGPContextRef		context,
	PGPKeyDBObjRef		theKey,
	PGPOptionListRef	optionList)
{
	PGPError			err;
	
	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( pgpOptionListIsValid( optionList ) );
	
	err = pgpGetOptionListError( optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpCheckOptionsInSet( optionList, sKeyOptionSet,
					elemsof( sKeyOptionSet ) );
		if( IsntPGPError( err ) )
		{
			CPGPKeyPassphraseDialogOptions	options;
			
			err = options.GatherOptions( context, optionList );
			if( IsntPGPError( err ) )
			{
				options.mDefaultKey	= theKey;	/* Overloading mDefaultKey */
				
				err = pgpKeyPassphraseDialogPlatform( context, &options );
			}
		}
	}
	
	return( err );
}

	PGPError
PGPKeyPassphraseDialog(
	PGPContextRef		context,
	PGPKeyDBObjRef		theKey,
	PGPOptionListRef	firstOption,
	...)
{
	PGPError	err;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( PGPKeyDBObjRefIsValid( theKey ) );

	if( pgpContextIsValid( context ) &&
		PGPKeyDBObjRefIsValid( theKey ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs(context, FALSE, firstOption, args);
		va_end( args );

		err = pgpKeyPassphraseDialog( context, theKey, optionList );
		
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	}
	
	return( err );
}

	static PGPError
FindValidSecretKey(
	PGPKeySetRef 	keySet,
	PGPBoolean			signing,
	PGPKeyDBObjRef	defaultPrivateKey,
	PGPKeyDBObjRef	preferredKey,
	PGPKeyDBObjRef	*secretKeyPtr)
{
	PGPError			err;
	PGPKeyListRef		keyList;
	PGPKeyDBObjProperty	keyProperty;
	PGPKeyDBObjRef		splitKeyRef 		= kInvalidPGPKeyDBObjRef;
	PGPKeyDBObjRef		secretKeyRef 		= kInvalidPGPKeyDBObjRef;
	PGPKeyDBObjRef		defaultKey			= kInvalidPGPKeyDBObjRef;
	PGPBoolean			foundDefaultKey		= FALSE;
	PGPBoolean			foundPreferredKey	= FALSE;
	PGPBoolean			preferredIsSplitKey	= FALSE;
	PGPBoolean			defaultIsSplitKey	= FALSE;
	
	*secretKeyPtr = kInvalidPGPKeyDBObjRef;
	
	/* Note that defaultKey may be a split key */
	if( ! PGPKeyDBObjRefIsValid( preferredKey ) )
		preferredKey = defaultPrivateKey;
		
	if( signing )
	{
		keyProperty = kPGPKeyProperty_CanSign;
	}
	else
	{
		keyProperty = kPGPKeyProperty_CanDecrypt;
	}
	
	err = PGPOrderKeySet( keySet, kPGPKeyOrdering_Any, FALSE, &keyList );
	if( IsntPGPError( err ) )
	{
		PGPKeyIterRef	keyIterator;
		
		err = PGPNewKeyIter( keyList, &keyIterator );
		if( IsntPGPError( err ) )
		{
			PGPKeyDBObjRef	theKey;
			
			err = PGPKeyIterNextKeyDBObj( keyIterator, kPGPKeyDBObjType_Key, &theKey );
			while( IsntPGPError( err ) )
			{
				PGPBoolean	canOperate;
				
				err = PGPGetKeyDBObjBooleanProperty( theKey, keyProperty, &canOperate );
				if( IsntPGPError( err ) )
				{
					if( canOperate )
					{
						PGPBoolean	isSplitKey = FALSE;
						
						(void) PGPGetKeyDBObjBooleanProperty( theKey,
									kPGPKeyProperty_IsSecretShared, &isSplitKey );

						if( theKey == defaultKey )
						{
							foundDefaultKey = TRUE;
							if( isSplitKey )
								defaultIsSplitKey = TRUE;
						}
						
						if( theKey == preferredKey )
						{
							foundPreferredKey = TRUE;
							if( isSplitKey )
								preferredIsSplitKey = TRUE;
						}
						
						if( isSplitKey )
						{
							if( ! PGPKeyDBObjRefIsValid( splitKeyRef ) )
								splitKeyRef = theKey;
						}
						else
						{
							if( ! PGPKeyDBObjRefIsValid( secretKeyRef ) )
								secretKeyRef = theKey;

							/*
							** If we've found the preferred key and we have
							** at least one non-split key, we can stop
							*/
							
							if( foundPreferredKey )
								break;
						}
					}
					
					err = PGPKeyIterNextKeyDBObj( keyIterator, kPGPKeyDBObjType_Key, &theKey );
				}
			}
			
			if( err == kPGPError_EndOfIteration )
				err = kPGPError_NoErr;
				
			PGPFreeKeyIter( keyIterator );
		}
		
		PGPFreeKeyList( keyList );
	}

	if( IsntPGPError( err ) )
	{
		if( PGPKeyDBObjRefIsValid( secretKeyRef ) )
		{
			/*
			** Do not allow a split key as the best secret key
			** if we're decrypting and at least one other secret
			** key is available.
			*/
			
			if( ! signing )
			{
				if( preferredIsSplitKey )
					foundPreferredKey = FALSE;

				if( defaultIsSplitKey )
					foundDefaultKey = FALSE;
			}
			
			if( foundPreferredKey )
			{
				*secretKeyPtr = preferredKey;
			}
			else if( foundDefaultKey )
			{
				*secretKeyPtr = defaultKey;
			}
			else
			{	
				*secretKeyPtr = secretKeyRef;
			}
		}
		else if( PGPKeyDBObjRefIsValid( splitKeyRef ) )
		{
			if( foundPreferredKey )
			{
				*secretKeyPtr = preferredKey;
			}
			else if( foundDefaultKey )
			{
				*secretKeyPtr = defaultKey;
			}
			else
			{	
				*secretKeyPtr = splitKeyRef;
			}
			
			if( signing )
			{
				err = kPGPError_KeyUnusableForSignature;
			}
			else
			{
				err = kPGPError_KeyUnusableForDecryption;
			}
		}
		else
		{
			err = kPGPError_SecretKeyNotFound;
		}
	}
	
	return( err );
}

static const PGPOptionType sSigningOptionSet[] =
{
	kCommonAllowedPassphraseOptions,
	kPGPOptionType_DefaultKey,
	kPGPOptionType_VerifyPassphrase,
	kPGPOptionType_FindMatchingKey
};

	static PGPError
pgpSigningPassphraseDialog(
	PGPContextRef		context,
	PGPKeyDBRef			allKeys,
	PGPOptionListRef	optionList,
	PGPKeyDBObjRef		*signingKey)
{
	PGPError	err;
	
	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( pgpOptionListIsValid( optionList ) );
	
	err = pgpGetOptionListError( optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpCheckOptionsInSet( optionList, sSigningOptionSet,
					elemsof( sSigningOptionSet ) );
		if( IsntPGPError( err ) )
		{
			CPGPSigningPassphraseDialogOptions	options;
			
			err = options.GatherOptions( context, optionList );
			if( IsntPGPError( err ) )
			{
				PGPKeySetRef	allKeysSet;
				
				err = PGPNewKeySet( allKeys, &allKeysSet );
				if( IsntPGPError( err ) )
				{
					err = FindValidSecretKey( allKeysSet, TRUE,
									options.mDefaultPrivateKey,
									options.mDefaultKey,
									&options.mDefaultKey );
					if( IsntPGPError( err ) )
					{
						options.mKeySet				= allKeysSet;
						options.mPassphraseKeyPtr	= signingKey;
						
						err = pgpSigningPassphraseDialogPlatform( context,
										&options );
					}
					else if( err == kPGPError_KeyUnusableForSignature )
					{
						/* Special case. We return a split key, if found */
						*signingKey = options.mDefaultKey;
					}
					
					PGPFreeKeySet( allKeysSet );
				}
			}
		}
	}
	
	return( err );
}

	PGPError
PGPSigningPassphraseDialog(
	PGPContextRef		context,
	PGPKeyDBRef 		allKeys,
	PGPKeyDBObjRef		*signingKey,
	PGPOptionListRef	firstOption,
	...)
{
	PGPError	err;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( PGPKeyDBRefIsValid( allKeys ) );

	if( IsntNull( signingKey ) )
		*signingKey = kInvalidPGPKeyDBObjRef;

	if( pgpContextIsValid( context ) &&
		PGPKeyDBRefIsValid( allKeys ) &&
		IsntNull( signingKey ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs(context, FALSE, firstOption, args);
		va_end( args );

		err = pgpSigningPassphraseDialog( context, allKeys,
						optionList, signingKey );
		
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	}
	
	return( err );
}

static const PGPOptionType sDecryptionOptionSet[] =
{
	kCommonAllowedPassphraseOptions,
	kPGPOptionType_DefaultKey,
	kPGPOptionType_VerifyPassphrase,
	kPGPOptionType_FindMatchingKey,
	kPGPOptionType_KeyServerUpdateParams
};

	static PGPError
pgpDecryptionPassphraseDialog(
	PGPContextRef		context,
	PGPKeySetRef		recipientKeys,
	const PGPKeyID		keyIDList[],
	PGPUInt32			keyIDCount,
	PGPOptionListRef	optionList,
	PGPKeyDBObjRef		*decryptionKey)
{
	PGPError	err;
	
	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( pgpOptionListIsValid( optionList ) );
	
	err = pgpGetOptionListError( optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpCheckOptionsInSet( optionList, sDecryptionOptionSet,
					elemsof( sDecryptionOptionSet ) );
		if( IsntPGPError( err ) )
		{
			CPGPDecryptionPassphraseDialogOptions	options;

			err = options.GatherOptions( context, optionList );
			if( IsntPGPError( err ) )
			{
				err = FindValidSecretKey( recipientKeys, FALSE,
							options.mDefaultPrivateKey, options.mDefaultKey,
							&options.mDefaultKey );
				
				/* Proceed with dialog even when no keys are found */
				if( err == kPGPError_SecretKeyNotFound )
					err = kPGPError_NoErr;
					
				if( IsntPGPError( err ) )
				{
					options.mKeySet				= recipientKeys;
					options.mPassphraseKeyPtr	= decryptionKey;
					options.mKeyIDList			= keyIDList;
					options.mKeyIDCount			= keyIDCount;
					
					err = options.GatherMissingKeys();
					if( IsntPGPError( err ) )
					{
						err = pgpDecryptionPassphraseDialogPlatform( context,
										&options );
					}
					
					if( IsPGPError( err ) &&
						IsntNull( options.mNewKeys ) &&
						PGPKeyDBRefIsValid( *options.mNewKeys ) )
					{
						PGPFreeKeyDB( *options.mNewKeys );
						*options.mNewKeys = kInvalidPGPKeyDBRef;
					}
				}
				else if( err == kPGPError_KeyUnusableForDecryption )
				{
					/* Special case. We return a split key, if found */
					*decryptionKey = options.mDefaultKey;
				}
			}
		}
	}
	
	return( err );
}

	PGPError
PGPDecryptionPassphraseDialog(
	PGPContextRef		context,
	PGPKeySetRef		recipientKeys,
	PGPUInt32			keyIDCount,
	const PGPKeyID		keyIDList[],
	PGPKeyDBObjRef		*decryptionKey,
	PGPOptionListRef	firstOption,
	...)
{
	PGPError	err = kPGPError_NoErr;
	va_list		args;
	
	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( PGPKeySetRefIsValid( recipientKeys ) );
	
	if( IsntNull( decryptionKey ) )
		*decryptionKey = kInvalidPGPKeyDBObjRef;

	if( pgpContextIsValid( context ) &&
		PGPKeySetRefIsValid( recipientKeys ) &&
		IsntNull( decryptionKey ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs(context, FALSE, firstOption, args);
		va_end( args );

		err = pgpDecryptionPassphraseDialog( context, recipientKeys, keyIDList,
						keyIDCount, optionList, decryptionKey );
		
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	}
	
	return( err );
}


static const PGPOptionType sConfirmationPassphraseOptionSet[] =
{
	kCommonAllowedPassphraseOptions,
	kPGPOptionType_ShowPassphraseQuality
};

	static PGPError
pgpConfirmationPassphraseDialog(
	PGPContextRef		context,
	PGPOptionListRef 	optionList,
	PGPUInt32			minPassphraseLength)
{
	PGPError	err;

	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( pgpOptionListIsValid( optionList ) );
	
	err = pgpGetOptionListError( optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpCheckOptionsInSet( optionList,
					sConfirmationPassphraseOptionSet,
					elemsof( sConfirmationPassphraseOptionSet ) );
		if( IsntPGPError( err ) )
		{
			CPGPConfirmationPassphraseDialogOptions	options;
			
			err = options.GatherOptions( context, optionList );
			if( IsntPGPError( err ) )
			{
				if( options.mMinPassphraseLength < minPassphraseLength )
				{
					options.mMinPassphraseLength = minPassphraseLength;
				}
					
				err = pgpConfirmationPassphraseDialogPlatform( context, 
									&options );
			}
		}
	}
	
	return( err );
}

	PGPError
PGPConfirmationPassphraseDialog(
	PGPContextRef		context,
	PGPOptionListRef	firstOption,
	...)
{
	PGPError	err = kPGPError_NoErr;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	
	if( pgpContextIsValid( context ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs(context, FALSE, firstOption, args);
		va_end( args );
	
		err = pgpConfirmationPassphraseDialog( context, optionList, 0 );
	
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	}
	
	return( err );
}

	PGPError
PGPConventionalEncryptionPassphraseDialog(
	PGPContextRef		context,
	PGPOptionListRef	firstOption,
	...)
{
	PGPError	err = kPGPError_NoErr;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	
	if( pgpContextIsValid( context ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs(context, FALSE, firstOption, args);
		va_end( args );
	
		err = pgpConfirmationPassphraseDialog( context, optionList, 0 );
	
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	}
	
	return( err );
}

static const PGPOptionType sRecipientOptionSet[] =
{
	kCommonAllowedOptions,
	kPGPOptionType_DialogPrompt,
	kPGPOptionType_DialogOptions,
	kPGPOptionType_DefaultRecipients,
	kPGPOptionType_DisplayMarginalValidity,
	kPGPOptionType_IgnoreMarginalValidity,
	kPGPOptionType_RecipientGroups,
	kPGPOptionType_ARREnforcement,
	kPGPOptionType_KeyServerUpdateParams,
	kPGPOptionType_RecipientList
};

	static PGPError
pgpRecipientDialog(
	PGPContextRef		context,
	PGPKeySetRef		allKeys,
	PGPBoolean			alwaysDisplay,
	PGPOptionListRef	optionList,
	PGPKeyDBRef 		*recipientKeys)
{
	PGPError			err;
	
	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( pgpOptionListIsValid( optionList ) );
	
	err = pgpGetOptionListError( optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpCheckOptionsInSet( optionList, sRecipientOptionSet,
					elemsof( sRecipientOptionSet ) );
		if( IsntPGPError( err ) )
		{
			CPGPRecipientDialogOptions	options;
			
			err = options.GatherOptions( context, optionList );
			if( IsntPGPError( err ) )
			{
				PGPKeyDBRef	workingKeyDB;
				
				// Copy the set of known keys into an in-memory DB so we can
				// make additions via key server searches
				
				err = PGPNewKeyDB( context, &workingKeyDB );
				if( IsntPGPError( err ) )
				{
					PGPKeySetRef	workingKeySet;
					
					err = PGPCopyKeys ( allKeys, workingKeyDB, &workingKeySet );
					if( IsntPGPError( err ) )
					{
						PGPKeyDBRef		allKeysDB = PGPPeekKeySetKeyDB( allKeys );
						
						err = PGPCheckKeyRingSigs( workingKeySet, allKeysDB, FALSE,
									NULL, 0 );
						if( IsntPGPError( err ) )
						{
							err = PGPCalculateTrust( workingKeySet, allKeysDB );
							
							if( IsntPGPError( err ) &&
								PGPKeyDBObjRefIsValid( options.mDefaultPrivateKey ) )
							{
								err = pgpFindKeyInKeyDB( options.mDefaultPrivateKey,
										workingKeyDB, &options.mDefaultPrivateKey );
							}
						}
					}
					
					if( IsntPGPError( err ) )
					{
						options.mClientKeySet		= workingKeySet;
						options.mAlwaysDisplay		= alwaysDisplay;
						options.mRecipientKeysPtr	= recipientKeys;
						
					#if HAVE_TEXT_UI
						if (options.mTextUI)
						{
							err = pgpRecipientDialogTTY(context,&options);
						}
						else
					#endif
						{
							err = pgpRecipientDialogPlatform(context,&options);
						}
												
						if( IsPGPError( err ) &&
							IsntNull( options.mNewKeys ) &&
							PGPKeyDBRefIsValid( *options.mNewKeys ) )
						{
							PGPFreeKeyDB( *options.mNewKeys );
							*options.mNewKeys = kInvalidPGPKeyDBRef;
						}
						
						PGPFreeKeySet( workingKeySet );
					}
					
					PGPFreeKeyDB( workingKeyDB );
				}
			}
		}
	}
	
	return( err );
}

	PGPError
PGPRecipientDialog(
	PGPContextRef		context,
	PGPKeyDBRef 		allKeys,
	PGPBoolean			alwaysDisplayDialog,
	PGPKeyDBRef 		*recipientKeys,
	PGPOptionListRef	firstOption,
	...)
{
	PGPError	err;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( PGPKeyDBRefIsValid( allKeys ) );
	pgpAssert( IsntNull( recipientKeys ) );

	if( pgpContextIsValid( context ) &&
		PGPKeyDBRefIsValid( allKeys ) &&
		IsntNull( recipientKeys ) )
	{
		PGPOptionListRef	optionList;
		PGPKeySetRef		allKeysSet;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context, FALSE, firstOption,
								args );
		va_end( args );

		err = PGPNewKeySet( allKeys, &allKeysSet );
		if( IsntPGPError( err ) )
		{
			err = pgpRecipientDialog( context, allKeysSet,
						alwaysDisplayDialog, optionList, recipientKeys );
			PGPFreeKeySet( allKeysSet );
		}
		
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	} 
	
	return( err );
}

static const PGPOptionType sRandomDataOptionSet[] =
{
	kCommonAllowedOptions,
	kPGPOptionType_DialogPrompt
};

	static PGPError
pgpCollectRandomDataDialog(
	PGPContextRef		context,
	PGPUInt32			neededEntropyBits,
	PGPOptionListRef	optionList)
{
	PGPError	err;
	
	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( pgpOptionListIsValid( optionList ) );
	
	err = pgpGetOptionListError( optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpCheckOptionsInSet( optionList, sRandomDataOptionSet,
					elemsof( sRandomDataOptionSet ) );
		if( IsntPGPError( err ) )
		{
			CPGPRandomDataDialogOptions	options;
			
			err = options.GatherOptions( context, optionList );
			if( IsntPGPError( err ) )
			{
				options.mNeededEntropyBits = neededEntropyBits;
				
				err = pgpCollectRandomDataDialogPlatform( context, &options );
			}
		}
	}
	
	return( err );
}

	PGPError
PGPCollectRandomDataDialog(
	PGPContextRef 		context,
	PGPUInt32			neededEntropyBits,
	PGPOptionListRef 	firstOption,
	... )
{
	PGPError	err;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	
	if( pgpContextIsValid( context ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context, FALSE, firstOption,
								args );
		va_end( args );

		err = pgpCollectRandomDataDialog( context, neededEntropyBits,
						optionList );
		
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	} 
	
	return( err );
}

static const PGPOptionType sSearchKeyServerOptionSet[] =
{
	kCommonAllowedKeyServerOptions,
	kPGPOptionType_KeyServerSearchFilter,
	kPGPOptionType_KeyServerSearchKeyIDList,
	kPGPOptionType_KeyServerSearchKey,
	kPGPOptionType_KeyServerSearchKeySet
};

#if PGP_WIN32
	PGPError
pgpCheckNetworklibAvailability(void)
{
	// We're simply static linking for now.
	return( kPGPError_NoErr );
}

#endif

	static PGPError
pgpSearchKeyServerDialog(
	PGPContextRef 			context,
	const PGPKeyServerSpec 	serverList[],
	PGPUInt32				serverCount,
	PGPtlsContextRef		tlsContext,
	PGPBoolean				searchAllServers,
	PGPKeyDBRef 			*foundKeys,
	PGPOptionListRef		optionList)
{
	PGPError	err;
	
	pgpAssert( pgpOptionListIsValid( optionList ) );
	
	err = pgpGetOptionListError( optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpCheckNetworklibAvailability();
		if( IsntPGPError( err ) )
		{
			err = pgpCheckOptionsInSet( optionList, sSearchKeyServerOptionSet,
						elemsof( sSearchKeyServerOptionSet ) );
			if( IsntPGPError( err ) )
			{
				CPGPSearchKeyServerDialogOptions	options;
				
				err = options.GatherOptions( context, optionList );
				if( IsntPGPError( err ) )
				{
					options.mServerList			= serverList;
					options.mServerCount		= serverCount;
					options.mTLSContext			= tlsContext;
					options.mSearchAllServers	= searchAllServers;
					options.mNewKeys			= foundKeys;
					
					err = pgpSearchKeyServerDialogPlatform(context, &options);
				}
			}
		}
	}
	
	return( err );
}

	PGPError
PGPSearchKeyServerDialog(
	PGPContextRef 			context,
	PGPUInt32				serverCount,
	const PGPKeyServerSpec 	serverList[],
	PGPtlsContextRef		tlsContext,
	PGPBoolean				searchAllServers,
	PGPKeyDBRef 			*foundKeys,
	PGPOptionListRef 		firstOption,
	... )
{
	PGPError	err;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( IsntNull( serverList ) );
	pgpAssert( serverCount >= 1 );
	pgpAssert( IsntNull( foundKeys ) );
	
	if( IsntNull( foundKeys ) )
		*foundKeys = kInvalidPGPKeyDBRef;
		
	if( pgpContextIsValid( context ) &&
		IsntNull( serverList ) &&
		serverCount >= 1 &&
		IsntNull( foundKeys ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context, FALSE, firstOption,
								args );
		va_end( args );

		err = pgpSearchKeyServerDialog( context, serverList, serverCount,
						tlsContext, searchAllServers, foundKeys, optionList );
		
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	} 
	
	return( err );
}

	PGPKeyDBObjRef
GetKeyForPassphrase(
	PGPKeySetRef	keySet,
	const char *	passphrase,
	PGPBoolean		signing)
{
	PGPKeyDBObjRef	theKey	= kInvalidPGPKeyDBObjRef;
	PGPError		err;
	PGPKeyListRef	keyListRef;
	PGPBoolean		foundValidKey	= FALSE;
	PGPBoolean		onToken;
	PGPUInt32		tryToken;
	
	err = PGPOrderKeySet( keySet, kPGPKeyOrdering_Any, FALSE, &keyListRef );
	if( IsntPGPError( err ) )
	{
		PGPKeyIterRef	keyIterator;
	
		/* Loop twice, first try non-token keys then token keys */
		for( tryToken=0; tryToken < 2; tryToken++ )
		{
			err = PGPNewKeyIter( keyListRef, &keyIterator );
			if( IsntPGPError( err ) )
			{
				err = PGPKeyIterNextKeyDBObj( keyIterator, kPGPKeyDBObjType_Key, &theKey );
				while( IsntPGPError( err ) )
				{
					PGPBoolean	tryKey = FALSE;

					if( signing )
					{
						PGPBoolean	canSign;

						if( IsntPGPError( PGPGetKeyDBObjBooleanProperty( theKey,
								kPGPKeyProperty_CanSign, &canSign ) ) && canSign )
						{
							tryKey = TRUE;
						}
					}
					else
					{
						PGPBoolean	canDecrypt;

						if( IsntPGPError( PGPGetKeyDBObjBooleanProperty( theKey,
								kPGPKeyProperty_CanDecrypt, &canDecrypt ) ) &&
									canDecrypt )
						{
							tryKey = TRUE;
						}
					}

					/*
					 * Danger with token keys: if we query them with
					 * bad passphrases (like passphrases for other
					 * keys) the token can lock up.  So we don't query
					 * them at all for signing keys, and we query them
					 * last for encryption keys.  (Note that signing
					 * test is done at the end of the tryToken loop.)
					 */
					if( tryKey
						&& IsntPGPError( PGPGetKeyDBObjBooleanProperty( theKey,
								kPGPKeyProperty_IsOnToken, &onToken ) ) )
					{
						tryKey = (!onToken == !tryToken);
					}
						
					if ( tryKey )
					{
						PGPContextRef	context	= PGPPeekKeyDBObjContext( theKey );

						if ( PGPPassphraseIsValid( theKey,
								PGPOPassphrase( context, passphrase),
								PGPOLastOption( context ) ) )
						{
							foundValidKey	= TRUE;
							break;
						}
					}

					err = PGPKeyIterNextKeyDBObj( keyIterator, kPGPKeyDBObjType_Key, &theKey );
				}

				PGPFreeKeyIter( keyIterator );
			}
			/* Don't try token keys if have a good one or if signing */
			if( foundValidKey || signing )
				break;
		}
		
		PGPFreeKeyList( keyListRef );
	}

	return( foundValidKey ? theKey : NULL );
}

static const PGPOptionType sSendToKeyServerOptionSet[] =
{
	kCommonAllowedKeyServerOptions
};

	static PGPError
pgpSendToKeyServerDialog(
	PGPContextRef 			context,
	const PGPKeyServerSpec 	*server,
	PGPtlsContextRef		tlsContext,
	PGPKeySetRef 			keysToSend,
	PGPKeySetRef 			*failedKeys,
	PGPOptionListRef		optionList)
{
	PGPError	err;
	
	pgpAssert( pgpOptionListIsValid( optionList ) );
	
	err = pgpGetOptionListError( optionList );
	if( IsntPGPError( err ) )
	{
		err = pgpCheckNetworklibAvailability();
		if( IsntPGPError( err ) )
		{
			err = pgpCheckOptionsInSet( optionList, sSendToKeyServerOptionSet,
						elemsof( sSendToKeyServerOptionSet ) );
			if( IsntPGPError( err ) )
			{
				CPGPSendToKeyServerDialogOptions	options;
				
				err = options.GatherOptions( context, optionList );
				if( IsntPGPError( err ) )
				{
					options.mServerList		= server;
					options.mServerCount	= 1;
					options.mTLSContext		= tlsContext;
					options.mKeysToSend		= keysToSend;
					options.mFailedKeys		= failedKeys;
					
					err = pgpSendToKeyServerDialogPlatform(context, &options);
				}
			}
		}
	}
	
	return( err );
}

	PGPError
PGPSendToKeyServerDialog(
	PGPContextRef 			context,
	const PGPKeyServerSpec 	*server,
	PGPtlsContextRef		tlsContext,
	PGPKeySetRef 			keysToSend,
	PGPKeySetRef 			*failedKeys,
	PGPOptionListRef 		firstOption,
	... )
{
	PGPError	err;
	va_list		args;

	pgpAssert( pgpContextIsValid( context ) );
	pgpAssert( IsntNull( server ) );
	pgpAssert( PGPKeySetRefIsValid( keysToSend ) );
	pgpAssert( IsntNull( failedKeys ) );
	
	if( IsntNull( failedKeys ) )
		*failedKeys = kInvalidPGPKeySetRef;
		
	if( pgpContextIsValid( context ) &&
		IsntNull( server ) &&
		PGPKeySetRefIsValid( keysToSend ) &&
		IsntNull( failedKeys ) )
	{
		PGPOptionListRef	optionList;
		
		va_start( args, firstOption );
		optionList = pgpBuildOptionListArgs( context, FALSE, firstOption,
								args );
		va_end( args );

		err = pgpSendToKeyServerDialog( context, server, tlsContext,
						keysToSend, failedKeys, optionList );
		
		PGPFreeOptionList( optionList );
	}
	else
	{
		va_start( args, firstOption );
		pgpFreeVarArgOptionList( firstOption, args );
		va_end( args );
		
		err = kPGPError_BadParams;
	} 
	
	return( err );
}
